//-----------------------------------------------------------------------------
// Torque 2D. 
// Copyright (C) GarageGames.com, Inc.
//-----------------------------------------------------------------------------



// --------------------------------------------------------------------
// Bit-Shifter Helper.
// --------------------------------------------------------------------
function BIT(%bitIndex)
{
	return 1<<%bitIndex;
}


//-----------------------------------------------------------------------------
// Add Mod-Token to front of mods list.
//-----------------------------------------------------------------------------
function pushFront(%list, %token, %delim)
{
	// Is the list empty?
	if (%list !$= "")
		// No, so add new token.
		return %token @ %delim @ %list;
		
	// Yes, so just use new token.
	return %token;
}

//-----------------------------------------------------------------------------
// Add Mod-Token to back of mods list.
//-----------------------------------------------------------------------------
function pushBack(%list, %token, %delim)
{
	// Is the list empty?
	if (%list !$= "")
		// No, so add new token.
		return %list @ %delim @ %token;
		
	// Yes, so just use new token.
	return %token;
}


//-----------------------------------------------------------------------------
// Remove Mod-Token from the mods list.
//-----------------------------------------------------------------------------
function popFront(%list, %delim)
{
	// Next Token.
	return nextToken(%list, unused, %delim);
}


//-----------------------------------------------------------------------------
// The displayHelp, onStart, onExit and parseArgs function are overriden
// by mod packages to get hooked into initialization and cleanup. 
//-----------------------------------------------------------------------------
function onStart()
{
   // Default startup function
}


//-----------------------------------------------------------------------------
// Exit Engine.
//-----------------------------------------------------------------------------
function onExit()
{
   // OnExit is called directly from C++ code, whereas onStart is
   // invoked at the end of this file.
}


//-----------------------------------------------------------------------------
// Parse Arguments.
//-----------------------------------------------------------------------------
function parseArgs()
{
   // Here for mod override, the arguments have already
   // been parsed.
}   


//-----------------------------------------------------------------------------
// Help Package.
//-----------------------------------------------------------------------------
package Help
{
   function onExit()
   {
      // Override onExit when displaying help
   }
};


//-----------------------------------------------------------------------------
// Default Help Display.
//-----------------------------------------------------------------------------
function displayHelp()
{
	// Activate the Help Package.
	activatePackage(Help);

	// Notes on logmode: console logging is written to console.log.
	// -log 0 disables console logging.
	// -log 1 appends to existing logfile; it also closes the file
	// (flushing the write buffer) after every write.
	// -log 2 overwrites any existing logfile; it also only closes
	// the logfile when the application shuts down.  (default)
	
	error(
	  "T2D command line options:\n"@
	  "  -log <logmode>         Logging behavior; see main.cs comments for details\n"@
	  "  -game <game_name>      Reset list of mods to only contain <game_name>\n"@
	  "  <game_name>            Works like the -game argument\n"@
	  "  -mod <mod_name>        Add <mod_name> to list of mods\n"@
	  "  -console               Open a separate console\n"@
	  "  -jSave  <file_name>    Record a journal\n"@
	  "  -jPlay  <file_name>    Play back a journal\n"@
	  "  -jDebug <file_name>    Play back a journal and issue an int3 at the end\n"@
	  "  -help                  Display this help message\n"
	);
}


//--------------------------------------------------------------------------
// Execute MOD Directory.
//--------------------------------------------------------------------------
function loadDir( %dir )
{
	// Set Mod Paths.
	setModPaths( pushback($userMods, %dir, ";") );
	
	// Execute Boot-strap file.
	exec( %dir @ "/main.cs" );
}


//--------------------------------------------------------------------------
// Execute startup scripts for each mod, starting at base and working up.
//--------------------------------------------------------------------------
function loadMods( %modPath )
{
	// Get MOD Path.
	%modPath = nextToken(%modPath, token, ";");
	
	// Did we get a MOD token?
	if (%modPath !$= "")
		// Yes, so recursively load it.
		loadMods(%modPath);

	// Execute MOD Boot-strap file.
	if( exec(%token @ "/main.cs") != true )
	{
		// Problem.
		error("Error: Unable to find specified mod: " @ %token );
		// Decrease MOD Count.
		$modcount--;
	}
}



//-----------------------------------------------------------------------------
//
// Before Release, set the "runWithEditors" option below to false and
// simply remove the appropriate editor directories from T2D.
//
//-----------------------------------------------------------------------------
$runWithEditors = false;
$modcount = 1;
$defaultGame = "TankPong";
$userMods = $defaultGame;
$displayHelp = false;

// Add Editors?
if ( $runWithEditors )
{
	// Yes!
	$userMods = "particleeditor;tileeditor;" @ $userMods;
	$modcount += 2;
}

//------------------------------------------------------------------------------
// Process command line arguments
//------------------------------------------------------------------------------
for ($i = 1; $i < $Game::argc ; $i++)
{
	// Set Current Argument.
	$arg = $Game::argv[$i];
	// Set Next Argument.
	$nextArg = $Game::argv[$i+1];
	// Check if we've got another argument.
	$hasNextArg = $Game::argc - $i > 1;
	// Log-Mode Off by default.
	$logModeSpecified = false;
	
	// Handle Argument Appropriately.
	switch$ ($arg)
	{		
		// ****************************************************
		// Console Logging.
		// ****************************************************
		case "-log":
		 $argUsed[$i]++;
		 if ($hasNextArg)
		 {
			// Turn on console logging
			if ($nextArg != 0)
			{
			   // Dump existing console to logfile first.
			   $nextArg += 4;
			}
			setLogMode($nextArg);
			$logModeSpecified = true;
			$argUsed[$i+1]++;
			$i++;
		 }
		 else
			error("Error: Missing Command Line argument. Usage: -log <Mode: 0,1,2>");
		
		
		// ****************************************************
		// MODs.
		// ****************************************************
		case "-mod":
		 $argUsed[$i]++;
		 if ($hasNextArg)
		 {
			// Append the mod to the end of the current list
			$userMods = strreplace($userMods, $nextArg, "");
			$userMods = pushFront($userMods, $nextArg, ";");
			$argUsed[$i+1]++;
			$i++;
			$modcount++;
		 }
		 else
			error("Error: Missing Command Line argument. Usage: -mod <mod_name>");

			
		// ****************************************************
		// GAME (Base or Complete Mod replacement).
		// ****************************************************
		case "-game":
		 $argUsed[$i]++;
		 if ($hasNextArg)
		 {
			// Remove all mods, start over with game
			$userMods = $nextArg;
			$argUsed[$i+1]++;
			$i++;
		$modcount = 1;
		 }
		 else
			error("Error: Missing Command Line argument. Usage: -game <game_name>");
		
		
		// ****************************************************
		// Console Mode. 
		// ****************************************************
		case "-console":
		 enableWinConsole(true);
		 $argUsed[$i]++;

		 
		// ****************************************************
		// Journal Save.
		// ****************************************************
		case "-jSave":
		 $argUsed[$i]++;
		 if ($hasNextArg)
		 {
			echo("Saving event log to journal: " @ $nextArg);
			saveJournal($nextArg);
			$argUsed[$i+1]++;
			$i++;
		 }
		 else
			error("Error: Missing Command Line argument. Usage: -jSave <journal_name>");

			
		// ****************************************************
		// Journal Play.
		// ****************************************************
		case "-jPlay":
		 $argUsed[$i]++;
		 if ($hasNextArg)
		 {
			playJournal($nextArg,false);
			$argUsed[$i+1]++;
			$i++;
		 }
		 else
			error("Error: Missing Command Line argument. Usage: -jPlay <journal_name>");

			
		// ****************************************************
		// Debugging.
		// ****************************************************
		case "-jDebug":
		 $argUsed[$i]++;
		 if ($hasNextArg)
		 {
			playJournal($nextArg,true);
			$argUsed[$i+1]++;
			$i++;
		 }
		 else
			error("Error: Missing Command Line argument. Usage: -jDebug <journal_name>");
		
		
		// ****************************************************
		// Command-Line Help.
		// ****************************************************
		case "-help":
		 $displayHelp = true;
		 $argUsed[$i]++;
		
		
		// ****************************************************
		// Default (Ignore).
		// ****************************************************
		default:
		 $argUsed[$i]++;
		 if($userMods $= "")
			$userMods = $arg;
	}
}


//--------------------------------------------------------------------------
// Do we have any mods?
//--------------------------------------------------------------------------
if( $modcount == 0 )
{
	// No, so reset user-mods.
	$userMods = $defaultGame;
	$modcount = 1;
}

//--------------------------------------------------------------------------
// Default to a new logfile each session.
//--------------------------------------------------------------------------
if ( !$logModeSpecified )
{
   setLogMode(6);
}

//--------------------------------------------------------------------------
// Set the mod path which dictates which directories will be visible
// to the scripts and the resource engine.
//--------------------------------------------------------------------------
setModPaths( $userMods );

//--------------------------------------------------------------------------
// Get the first mod on the list, which will be the last to be applied... this
// does not modify the list.
//--------------------------------------------------------------------------
nextToken( $userMods, currentMod, ";" );

//--------------------------------------------------------------------------
// Start the MOD loading process...
//--------------------------------------------------------------------------
echo( "--------- Loading MODS ---------" );
loadMods( $userMods );
echo( "" );

// Are there any mods?
if( $modcount == 0 )
{
	// No, so enable console.
	enableWinConsole(true);
	// Show Error.
	error("Error: Unable to load any specified mods");
	// Quit!
	quit();	
}

//--------------------------------------------------------------------------
// Parse the command line arguments.
//--------------------------------------------------------------------------
echo("--------- Parsing Arguments ---------");
parseArgs();


//--------------------------------------------------------------------------
// Either display the help message or startup the application.
//--------------------------------------------------------------------------

// Are we displaying help?
if ( $displayHelp )
{
	// Yes, so enable console.
	enableWinConsole(true);
	// Display Help.
	displayHelp();
	// Quit!
	quit();
}
else
{
	// Start-up the script mods.
	onStart();
	// User Info.
	echo("T2D Engine initialized...");
}

//--------------------------------------------------------------------------
// Display an error message for unused arguments
//--------------------------------------------------------------------------
for ($i = 1; $i < $Game::argc; $i++)
{
	// Unused?
	if ( !$argUsed[$i] )
		// Yes, so show error.
		error("Error: Unknown command line argument: " @ $Game::argv[$i]);
}

